<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ARC_Logger {

    private $initial_request_uri;

    public function __construct() {
        add_action('init', [$this, 'capture_initial_request_uri'], 0);
        // Rimosso l'aggancio a wp_redirect per evitare interferenze.
        // Ora ci agganciamo al hook specifico di Rank Math per i redirect.
        add_action('rank_math/redirection/add_hit', [$this, 'log_rank_math_redirect_hit'], 10, 4);
    }

    /**
     * Cattura l'URI della richiesta iniziale.
     */
    public function capture_initial_request_uri() {
        $this->initial_request_uri = $_SERVER['REQUEST_URI'];
    }

    /**
     * Logga i dettagli del redirect di Rank Math nel database.
     * Questo hook viene chiamato da Rank Math dopo che un redirect è stato eseguito con successo.
     *
     * @param int    $redirection_id L'ID del redirect di Rank Math.
     * @param string $old_url        L'URL originale richiesto.
     * @param string $new_url        L'URL di destinazione del redirect.
     * @param int    $status_code    Il codice di stato HTTP del redirect.
     */
    public function log_rank_math_redirect_hit( $redirection_id, $old_url, $new_url, $status_code ) {
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $referrer = isset($_SERVER['HTTP_REFERER']) ? esc_url_raw(wp_unslash($_SERVER['HTTP_REFERER'])) : '';

        // Log solo se old_url e new_url sono diversi per evitare di loggare auto-redirect o canonical che non sono veri redirect
        if (trailingslashit($old_url) !== trailingslashit($new_url)) {
            ARC_DB::insert_log_entry( $old_url, $new_url, $status_code, $user_agent, $referrer );
        }
    }
}