<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ARC_DB {

    private static $table_name;

    public static function init() {
        global $wpdb;
        self::$table_name = $wpdb->prefix . ARC_REDIRECT_LOG_TABLE;
    }

    /**
     * Creates the database table for redirect logs.
     */
    public static function create_log_table() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE " . self::$table_name . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            old_url varchar(2048) NOT NULL,
            new_url varchar(2048) NOT NULL,
            status_code int(11) NOT NULL,
            access_time datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            user_agent varchar(255) DEFAULT '' NOT NULL,
            referrer varchar(2048) DEFAULT '' NOT NULL,
            PRIMARY_KEY  (id),
            KEY old_url (old_url(255))
        ) $charset_collate;";

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );
    }

    /**
     * Drops the database table for redirect logs.
     */
    public static function drop_log_table() {
        global $wpdb;
        $wpdb->query( "DROP TABLE IF EXISTS " . self::$table_name );
    }

    /**
     * Inserts a new redirect log entry.
     *
     * @param string $old_url The URL that was requested.
     * @param string $new_url The URL the request was redirected to.
     * @param int    $status_code The HTTP status code of the redirect (e.g., 301, 302).
     * @param string $user_agent The user agent of the client.
     * @param string $referrer The referrer URL.
     */
    public static function insert_log_entry( $old_url, $new_url, $status_code, $user_agent = '', $referrer = '' ) {
        global $wpdb;
        $result = $wpdb->insert(
            self::$table_name,
            array(
                'old_url'     => $old_url,
                'new_url'     => $new_url,
                'status_code' => $status_code,
                'user_agent'  => $user_agent,
                'referrer'    => $referrer,
            ),
            array(
                '%s', // old_url
                '%s', // new_url
                '%d', // status_code
                '%s', // user_agent
                '%s', // referrer
            )
        );

        if (false === $result) {
            // Log the error to WordPress debug log
            error_log('ARC_DB Error: Failed to insert redirect log entry. WPDB Error: ' . $wpdb->last_error);
            error_log('ARC_DB Data: old_url=' . $old_url . ', new_url=' . $new_url . ', status_code=' . $status_code);
        }
        return $result;
    }

    /**
     * Retrieves redirect logs from the database with pagination and optional search.
     *
     * @param array $args Query arguments.
     * @return array List of redirect log objects.
     */
    public static function get_redirect_logs( $args = [] ) {
        global $wpdb;
        $defaults = [
            'limit'  => 10, // Default limit for pagination
            'offset' => 0,
            'order_by' => 'access_time',
            'order' => 'DESC',
            'search' => '',
        ];
        $args = wp_parse_args( $args, $defaults );

        // Sanitize order_by and order to prevent SQL injection
        $order_by = sanitize_sql_orderby($args['order_by']);
        $order = (strtoupper($args['order']) === 'ASC') ? 'ASC' : 'DESC';

        $where_clauses = [];
        $query_params = [];

        if (!empty($args['search'])) {
            $where_clauses[] = "(old_url LIKE %s OR new_url LIKE %s)";
            $query_params[] = '%' . $wpdb->esc_like($args['search']) . '%';
            $query_params[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }

        $where_sql = count($where_clauses) > 0 ? ' WHERE ' . implode(' AND ', $where_clauses) : '';

        $query = $wpdb->prepare(
            "SELECT * FROM " . self::$table_name . $where_sql . " ORDER BY {$order_by} {$order} LIMIT %d OFFSET %d",
            array_merge($query_params, [$args['limit'], $args['offset']])
        );

        return $wpdb->get_results( $query );
    }

    /**
     * Counts total redirect logs with optional search.
     *
     * @param array $args Query arguments.
     * @return int Total number of logs.
     */
    public static function count_redirect_logs( $args = [] ) {
        global $wpdb;
        $defaults = [
            'search' => '',
        ];
        $args = wp_parse_args( $args, $defaults );

        $where_clauses = [];
        $query_params = [];

        if (!empty($args['search'])) {
            $where_clauses[] = "(old_url LIKE %s OR new_url LIKE %s)";
            $query_params[] = '%' . $wpdb->esc_like($args['search']) . '%';
            $query_params[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }

        $where_sql = count($where_clauses) > 0 ? ' WHERE ' . implode(' AND ', $where_clauses) : '';

        $query = "SELECT COUNT(id) FROM " . self::$table_name . $where_sql;
        return (int) $wpdb->get_var( $wpdb->prepare($query, $query_params) );
    }

    /**
     * Deletes a specific redirect log entry by ID.
     *
     * @param int $log_id The ID of the log entry to delete.
     * @return bool True on success, false on failure.
     */
    public static function delete_log_entry( $log_id ) {
        global $wpdb;
        return $wpdb->delete(
            self::$table_name,
            array( 'id' => $log_id ),
            array( '%d' )
        );
    }

    /**
     * Retrieves the most accessed redirect URLs from the logs within a given timeframe.
     *
     * @param int $limit The number of top redirects to retrieve.
     * @param int $days The number of days to look back.
     * @return array List of objects with old_url and access_count.
     */
    public static function get_most_accessed_redirects( $limit = 5, $days = 30 ) {
        global $wpdb;
        $query = $wpdb->prepare(
            "SELECT old_url, COUNT(id) as access_count FROM " . self::$table_name . " WHERE access_time >= DATE_SUB(NOW(), INTERVAL %d DAY) GROUP BY old_url ORDER BY access_count DESC LIMIT %d",
            $days,
            $limit
        );
        return $wpdb->get_results( $query );
    }

    /**
     * Retrieves the least accessed redirect URLs from the logs within a given timeframe.
     * This will include URLs with non-zero access counts, effectively showing "rarely used".
     *
     * @param int $limit The number of least redirects to retrieve.
     * @param int $days The number of days to look back.
     * @return array List of objects with old_url and access_count.
     */
    public static function get_least_accessed_redirects( $limit = 5, $days = 30 ) {
        global $wpdb;
        $query = $wpdb->prepare(
            "SELECT old_url, COUNT(id) as access_count FROM " . self::$table_name . " WHERE access_time >= DATE_SUB(NOW(), INTERVAL %d DAY) GROUP BY old_url HAVING access_count > 0 ORDER BY access_count ASC LIMIT %d",
            $days,
            $limit
        );
        return $wpdb->get_results( $query );
    }

    /**
     * Ottiene l'ultimo orario di accesso per un dato URL.
     * @param string $url L'URL da cercare.
     * @return string|null L'orario dell'ultimo accesso o null se non trovato.
     */
    public static function get_last_access_time_for_url($url) {
        global $wpdb;
        $table = self::$table_name;
        $query = $wpdb->prepare(
            "SELECT access_time FROM $table WHERE old_url = %s ORDER BY access_time DESC LIMIT 1",
            $url
        );
        return $wpdb->get_var($query);
    }
}