<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ARC_Analyzer {

    public function __construct() {
        // Nessuna azione immediata nel costruttore per ora
    }

    /**
     * Recupera i redirect configurati in Rank Math, inclusi i conteggi delle hit.
     * @param string $search Termine di ricerca opzionale.
     * @return array Lista di redirect di Rank Math.
     */
    public function get_rank_math_redirects($search = '') {
        global $wpdb;
        $table = $wpdb->prefix . 'rank_math_redirections';

        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") == $table) {
            $where_clauses = [];
            $query_params = [];

            if (!empty($search)) {
                $where_clauses[] = "(url_from LIKE %s OR url_to LIKE %s)";
                $query_params[] = '%' . $wpdb->esc_like($search) . '%';
                $query_params[] = '%' . $wpdb->esc_like($search) . '%';
            }

            $where_sql = count($where_clauses) > 0 ? ' WHERE ' . implode(' AND ', $where_clauses) : '';
            $query = "SELECT * FROM $table" . $where_sql;

            return $wpdb->get_results($wpdb->prepare($query, $query_params));
        }
        return [];
    }

    /**
     * Recupera i log dei redirect dal database del plugin con paginazione e ricerca.
     * @param array $args Argomenti per la query (es. 'limit', 'offset', 'order_by', 'order', 'search').
     * @return array Lista di log dei redirect.
     */
    public function get_logged_redirects( $args = [] ) {
        return ARC_DB::get_redirect_logs( $args );
    }

    /**
     * Esegue l'analisi dei redirect per identificare inutilizzati, duplicati, ecc.
     *
     * @param int $log_limit Limite per i log da recuperare per l'analisi.
     * @param string $search Termine di ricerca opzionale.
     * @return array Risultati dell'analisi.
     */
    public function analyze_redirects( $log_limit = 1000, $search = '' ) { // Aumentato il limite per un'analisi più completa
        $rank_math_redirects = $this->get_rank_math_redirects($search);
        $logged_redirects_all = ARC_DB::get_redirect_logs(['limit' => $log_limit, 'order' => 'ASC', 'search' => $search]); // Recupera più log per l'analisi

        // Mappa i log del nostro plugin per un accesso più rapido e per l'ultimo accesso
        $logged_urls = [];
        $last_access_times = [];
        foreach ($logged_redirects_all as $log) {
            $logged_urls[trailingslashit($log->old_url)] = true;
            if (!isset($last_access_times[$log->old_url]) || $log->access_time > $last_access_times[$log->old_url]) {
                $last_access_times[$log->old_url] = $log->access_time;
            }
        }

        // Identificare redirect di Rank Math mai usati e quelli che puntano alla homepage
        $unused_redirects = [];
        $homepage_redirects = [];
        $site_url = trailingslashit(home_url());

        foreach ($rank_math_redirects as $rm_redirect) {
            // Assicurati che la proprietà 'hits' esista, altrimenti usa 0
            $rm_redirect_hits = property_exists($rm_redirect, 'hits') ? (int) $rm_redirect->hits : 0;
            $from_url_slashed = trailingslashit($rm_redirect->url_from);

            // Un redirect è considerato inutilizzato se:
            // 1. Il suo 'hits' in Rank Math è 0
            // E
            // 2. Non è stato loggato dal nostro plugin
            if ( $rm_redirect_hits === 0 && !isset($logged_urls[$from_url_slashed]) ) {
                $unused_redirects[] = $rm_redirect;
            }

            // Check for redirects to homepage
            $to_url_slashed = property_exists($rm_redirect, 'url_to') ? trailingslashit($rm_redirect->url_to) : '';
            if ($to_url_slashed === $site_url) {
                $homepage_redirects[] = $rm_redirect;
            }
        }

        return [
            'rank_math_redirects' => $rank_math_redirects,
            'logged_redirects_all' => $logged_redirects_all, // Tutti i log per analisi interna
            'unused_redirects' => $unused_redirects,
            'homepage_redirects' => $homepage_redirects,
            'most_accessed_redirects' => ARC_DB::get_most_accessed_redirects(5, 30), // Ultimi 30 giorni
            'least_accessed_redirects' => ARC_DB::get_least_accessed_redirects(5, 30), // Ultimi 30 giorni, non zero
            'last_access_times' => $last_access_times, // Passa i tempi di ultimo accesso dal nostro log
        ];
    }
}