<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class ARC_Admin {
    private $analyzer;
    private $logs_per_page = 10;

    public function __construct() {
        add_action('admin_menu', [$this, 'add_admin_page']);
        add_action('admin_post_arc_delete_log', [$this, 'handle_delete_log']);
        add_action('admin_post_arc_export_logs', [$this, 'handle_export_logs']);
        add_action('admin_post_arc_deactivate_redirect', [$this, 'handle_deactivate_redirect']); // Nuovo handler per disattivare
        $this->analyzer = new ARC_Analyzer();
    }

    public function add_admin_page() {
        add_menu_page(
            'Auto Redirect Cleaner',
            'Redirect Cleaner',
            'manage_options',
            'arc-dashboard',
            [$this, 'render_dashboard'],
            'dashicons-randomize',
            60 // Posizione nel menu
        );
    }

    /**
     * Handles the deletion of a redirect log entry.
     */
    public function handle_delete_log() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'You do not have sufficient permissions to access this page.' );
        }

        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'arc_delete_log_nonce' ) ) {
            wp_die( 'Security check failed.' );
        }

        $log_id = isset( $_POST['log_id'] ) ? intval( $_POST['log_id'] ) : 0;

        if ( $log_id > 0 ) {
            if ( ARC_DB::delete_log_entry( $log_id ) ) {
                wp_redirect( admin_url( 'admin.php?page=arc-dashboard&message=log_deleted' ) );
                exit;
            } else {
                wp_redirect( admin_url( 'admin.php?page=arc-dashboard&message=log_delete_failed' ) );
                exit;
            }
        }
        wp_redirect( admin_url( 'admin.php?page=arc-dashboard' ) );
        exit;
    }

    /**
     * Handles the export of redirect logs to CSV.
     */
    public function handle_export_logs() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'You do not have sufficient permissions to access this page.' );
        }

        // Nonce verification for GET request
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'arc_export_logs_nonce' ) ) {
            wp_die( 'Security check failed.' );
        }

        $logs = ARC_DB::get_redirect_logs(['limit' => 999999]); // Get all logs

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=redirect_logs_' . date('Y-m-d') . '.csv');

        $output = fopen('php://output', 'w');

        // CSV Header
        fputcsv($output, ['ID', 'Old URL', 'New URL', 'Status Code', 'Access Time', 'User Agent', 'Referrer']);

        // CSV Data
        foreach ($logs as $log) {
            fputcsv($output, [
                $log->id,
                $log->old_url,
                $log->new_url,
                $log->status_code,
                $log->access_time,
                $log->user_agent,
                $log->referrer,
            ]);
        }

        fclose($output);
        exit;
    }

    /**
     * Handles the deactivation of a Rank Math redirect.
     */
    public function handle_deactivate_redirect() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( 'You do not have sufficient permissions to access this page.' );
        }

        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'arc_deactivate_redirect_nonce' ) ) {
            wp_die( 'Security check failed.' );
        }

        $redirect_id = isset( $_POST['redirect_id'] ) ? intval( $_POST['redirect_id'] ) : 0;

        if ( $redirect_id > 0 ) {
            // Check if Rank Math Redirection class exists
            if ( class_exists('RankMath\\Redirections\\Redirection') ) {
                $updated = \RankMath\Redirections\Redirection::update(
                    [
                        'id' => $redirect_id,
                        'status' => 'inactive',
                    ]
                );

                if ( $updated ) {
                    wp_redirect( admin_url( 'admin.php?page=arc-dashboard&message=redirect_deactivated' ) );
                    exit;
                } else {
                    wp_redirect( admin_url( 'admin.php?page=arc-dashboard&message=redirect_deactivate_failed' ) );
                    exit;
                }
            } else {
                wp_redirect( admin_url( 'admin.php?page=arc-dashboard&message=rank_math_not_found' ) );
                exit;
            }
        }
        wp_redirect( admin_url( 'admin.php?page=arc-dashboard' ) );
        exit;
    }

    public function render_dashboard() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Handle messages
        if ( isset( $_GET['message'] ) ) {
            if ( $_GET['message'] === 'log_deleted' ) {
                echo '<div class="notice notice-success is-dismissible"><p>Log entry deleted successfully!</p></div>';
            } elseif ( $_GET['message'] === 'log_delete_failed' ) {
                echo '<div class="notice notice-error is-dismissible"><p>Failed to delete log entry.</p></div>';
            } elseif ( $_GET['message'] === 'redirect_deactivated' ) {
                echo '<div class="notice notice-success is-dismissible"><p>Redirect di Rank Math disattivato con successo!</p></div>';
            } elseif ( $_GET['message'] === 'redirect_deactivate_failed' ) {
                echo '<div class="notice notice-error is-dismissible"><p>Errore durante la disattivazione del redirect di Rank Math.</p></div>';
            } elseif ( $_GET['message'] === 'rank_math_not_found' ) {
                echo '<div class="notice notice-error is-dismissible"><p>Classe di gestione redirect di Rank Math non trovata. Assicurati che Rank Math sia attivo e aggiornato.</p></div>';
            }
        }

        $search_query = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';

        // Recupera i dati dall'analizzatore
        $analysis_results = $this->analyzer->analyze_redirects(1000, $search_query);
        $rank_math_redirects = $analysis_results['rank_math_redirects'];
        $unused_redirects = $analysis_results['unused_redirects'];
        $homepage_redirects = $analysis_results['homepage_redirects'];
        $most_accessed_redirects = $analysis_results['most_accessed_redirects'];
        $least_accessed_redirects = $analysis_results['least_accessed_redirects'];
        $last_access_times = $analysis_results['last_access_times'];

        $total_rank_math_redirects = count($rank_math_redirects);
        $total_logged_redirects = ARC_DB::count_redirect_logs(['search' => $search_query]);
        $total_unused_redirects = count($unused_redirects);
        $total_homepage_redirects = count($homepage_redirects);

        // Pagination for logged redirects
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($current_page - 1) * $this->logs_per_page;
        $logged_redirects_paginated = ARC_DB::get_redirect_logs([
            'limit' => $this->logs_per_page,
            'offset' => $offset,
            'order_by' => 'access_time',
            'order' => 'DESC',
            'search' => $search_query,
        ]);
        $total_log_pages = ceil($total_logged_redirects / $this->logs_per_page);

        ?>
        <div class="wrap">
            <h1>Auto Redirect Cleaner Dashboard</h1>
            <p>Benvenuto nella dashboard di Auto Redirect Cleaner. Qui puoi analizzare e ottimizzare i tuoi redirect.</p>

            <?php if ($total_unused_redirects > 0) : ?>
                <div class="notice notice-warning is-dismissible">
                    <p><strong>⚠️ Attenzione:</strong> Sono stati rilevati <?php echo esc_html($total_unused_redirects); ?> redirect di Rank Math che non sono mai stati usati. Considera di disattivarli o eliminarli per migliorare le prestazioni e la SEO.</p>
                </div>
            <?php endif; ?>

            <h2 class="title">Statistiche Generali</h2>
            <div class="arc-stats-cards">
                <div class="arc-card">
                    <div class="arc-card-icon" style="background-color: #e0f7fa;"><span class="dashicons dashicons-admin-links" style="color: #00bcd4;"></span></div>
                    <div class="arc-card-content">
                        <p class="arc-card-title">Totale Redirect Rank Math</p>
                        <p class="arc-card-number"><?php echo esc_html($total_rank_math_redirects); ?></p>
                    </div>
                </div>
                <div class="arc-card">
                    <div class="arc-card-icon" style="background-color: #e8f5e9;"><span class="dashicons dashicons-chart-bar" style="color: #4caf50;"></span></div>
                    <div class="arc-card-content">
                        <p class="arc-card-title">Accessi Loggati</p>
                        <p class="arc-card-number"><?php echo esc_html($total_logged_redirects); ?></p>
                    </div>
                </div>
                <div class="arc-card">
                    <div class="arc-card-icon" style="background-color: #fff3e0;"><span class="dashicons dashicons-warning" style="color: #ff9800;"></span></div>
                    <div class="arc-card-content">
                        <p class="arc-card-title">Redirect Inutilizzati (Stimati)</p>
                        <p class="arc-card-number"><?php echo esc_html($total_unused_redirects); ?></p>
                    </div>
                </div>
                <div class="arc-card">
                    <div class="arc-card-icon" style="background-color: #ffebee;"><span class="dashicons dashicons-admin-home" style="color: #f44336;"></span></div>
                    <div class="arc-card-content">
                        <p class="arc-card-title">Redirect a Homepage</p>
                        <p class="arc-card-number"><?php echo esc_html($total_homepage_redirects); ?></p>
                    </div>
                </div>
            </div>

            <h2 class="title">Redirect di Rank Math</h2>
            <div class="postbox">
                <div class="inside">
                    <div class="arc-search-bar">
                        <form method="get" action="<?php echo esc_url(admin_url('admin.php')); ?>">
                            <input type="hidden" name="page" value="arc-dashboard" />
                            <input type="text" name="s" value="<?php echo esc_attr($search_query); ?>" placeholder="Cerca URL..." class="regular-text" />
                            <input type="submit" class="button" value="Cerca" />
                            <?php if (!empty($search_query)) : ?>
                                <a href="<?php echo esc_url(admin_url('admin.php?page=arc-dashboard')); ?>" class="button">Reset</a>
                            <?php endif; ?>
                        </form>
                    </div>
                    <?php if (empty($rank_math_redirects)) : ?>
                        <p>Nessun redirect di Rank Math trovato.</p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped arc-sticky-header">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>URL da</th>
                                    <th>URL a</th>
                                    <th>Tipo</th>
                                    <th>Stato RM</th>
                                    <th>Hit RM</th>
                                    <th>Stato Analisi</th>
                                    <th>Ultimo Accesso (Plugin)</th>
                                    <th>Azioni</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $unused_urls = array_map(function($r) { return trailingslashit($r->url_from); }, $unused_redirects);
                                foreach ($rank_math_redirects as $redirect) :
                                    $from_url_slashed = trailingslashit($redirect->url_from);
                                    $analysis_status = 'Usato';
                                    $analysis_status_class = 'arc-badge-success';
                                    $deactivate_button = '';

                                    // Assicurati che la proprietà 'hits' esista, altrimenti usa 0
                                    $rm_redirect_hits = property_exists($redirect, 'hits') ? (int) $redirect->hits : 0;

                                    // Un redirect è considerato inutilizzato se:
                                    // 1. Il suo 'hits' in Rank Math è 0
                                    // E
                                    // 2. Non è stato loggato dal nostro plugin
                                    if ( $rm_redirect_hits === 0 && !isset($last_access_times[$redirect->url_from]) ) {
                                        $analysis_status = 'Inutilizzato';
                                        $analysis_status_class = 'arc-badge-warning';
                                        if ($redirect->status === 'active') { // Mostra il pulsante solo se il redirect è attivo
                                            $deactivate_button = '<form method="post" action="' . esc_url( admin_url('admin-post.php') ) . '" style="display:inline-block; margin-left: 10px;">
                                                <input type="hidden" name="action" value="arc_deactivate_redirect" />
                                                <input type="hidden" name="redirect_id" value="' . esc_attr($redirect->id) . '" />
                                                ' . wp_nonce_field( 'arc_deactivate_redirect_nonce', '_wpnonce', true, false ) . '
                                                <button type="submit" class="button button-small" title="Disattiva questo redirect in Rank Math" onclick="return confirm(\'Sei sicuro di voler disattivare questo redirect in Rank Math?\');"><span class="dashicons dashicons-no"></span></button>
                                            </form>';
                                        }
                                    }

                                    $last_access = isset($last_access_times[$redirect->url_from]) ? $last_access_times[$redirect->url_from] : 'N/D';
                                    if ($last_access === 'N/D' && $rm_redirect_hits > 0) {
                                        $last_access = 'N/D (Hit RM: ' . esc_html($rm_redirect_hits) . ')';
                                    }
                                ?>
                                    <tr>
                                        <td><?php echo esc_html($redirect->id); ?></td>
                                        <td>
                                            <span title="<?php echo esc_attr($redirect->url_from); ?>"><?php echo esc_html(wp_basename($redirect->url_from)); ?></span>
                                            <button class="button-link arc-copy-url" data-url="<?php echo esc_attr($redirect->url_from); ?>" title="Copia URL"><span class="dashicons dashicons-admin-page"></span></button>
                                        </td>
                                        <td><span title="<?php echo esc_attr($redirect->url_to); ?>"><?php echo esc_html(wp_basename($redirect->url_to)); ?></span></td>
                                        <td><span class="arc-badge arc-badge-info" title="Tipo di redirect"><?php echo esc_html($redirect->type); ?></span></td>
                                        <td><span class="arc-badge arc-badge-<?php echo esc_attr($redirect->status === 'active' ? 'success' : 'danger'); ?>" title="Stato del redirect in Rank Math"><?php echo esc_html(ucfirst($redirect->status)); ?></span></td>
                                        <td><?php echo esc_html($rm_redirect_hits); ?></td>
                                        <td><span class="arc-badge <?php echo esc_attr($analysis_status_class); ?>" title="Stato di analisi del plugin"><?php echo esc_html($analysis_status); ?></span></td>
                                        <td><?php echo esc_html($last_access); ?></td>
                                        <td><?php echo $deactivate_button; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>

            <?php if ($total_homepage_redirects > 0) : ?>
            <h2 class="title">Redirect di Rank Math che puntano alla Homepage</h2>
            <div class="postbox">
                <div class="inside">
                    <p>Questi redirect puntano direttamente alla homepage. In alcuni casi, potrebbe essere preferibile reindirizzare a una pagina più specifica o gestire la risorsa originale.</p>
                    <table class="wp-list-table widefat fixed striped arc-sticky-header">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>URL da</th>
                                <th>URL a</th>
                                <th>Tipo</th>
                                <th>Stato RM</th>
                                <th>Hit RM</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($homepage_redirects as $redirect) : ?>
                                <?php
                                    $rm_redirect_hits = property_exists($redirect, 'hits') ? (int) $redirect->hits : 0;
                                ?>
                                <tr>
                                    <td><?php echo esc_html($redirect->id); ?></td>
                                    <td>
                                        <span title="<?php echo esc_attr($redirect->url_from); ?>"><?php echo esc_html(wp_basename($redirect->url_from)); ?></span>
                                        <button class="button-link arc-copy-url" data-url="<?php echo esc_attr($redirect->url_from); ?>" title="Copia URL"><span class="dashicons dashicons-admin-page"></span></button>
                                    </td>
                                    <td><span title="<?php echo esc_attr($redirect->url_to); ?>"><?php echo esc_html(wp_basename($redirect->url_to)); ?></span></td>
                                    <td><span class="arc-badge arc-badge-info" title="Tipo di redirect"><?php echo esc_html($redirect->type); ?></span></td>
                                    <td><span class="arc-badge arc-badge-<?php echo esc_attr($redirect->status === 'active' ? 'success' : 'danger'); ?>" title="Stato del redirect in Rank Math"><?php echo esc_html(ucfirst($redirect->status)); ?></span></td>
                                    <td><?php echo esc_html($rm_redirect_hits); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
            <?php endif; ?>

            <h2 class="title">Log degli Accessi ai Redirect</h2>
            <div class="postbox">
                <div class="inside">
                    <div style="margin-bottom: 15px;">
                        <a href="<?php echo esc_url( wp_nonce_url( admin_url('admin-post.php?action=arc_export_logs'), 'arc_export_logs_nonce', '_wpnonce' ) ); ?>" class="button button-primary">
                            <span class="dashicons dashicons-download"></span> Esporta Log CSV
                        </a>
                    </div>

                    <?php if (empty($logged_redirects_paginated)) : ?>
                        <p>Nessun accesso ai redirect loggato finora.</p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped arc-sticky-header">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>URL Richiesto</th>
                                    <th>URL Destinazione</th>
                                    <th>Status</th>
                                    <th>Data/ora <span title="Fuso orario del server">(<?php echo esc_html(wp_timezone_string()); ?>)</span></th>
                                    <th>User Agent</th>
                                    <th>Referrer</th>
                                    <th>Azioni</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($logged_redirects_paginated as $log) : ?>
                                    <tr>
                                        <td><?php echo esc_html($log->id); ?></td>
                                        <td>
                                            <span title="<?php echo esc_attr($log->old_url); ?>"><?php echo esc_html(wp_basename($log->old_url)); ?></span>
                                            <button class="button-link arc-copy-url" data-url="<?php echo esc_attr($log->old_url); ?>" title="Copia URL"><span class="dashicons dashicons-admin-page"></span></button>
                                        </td>
                                        <td><span title="<?php echo esc_attr($log->new_url); ?>"><?php echo esc_html(wp_basename($log->new_url)); ?></span></td>
                                        <td><span class="arc-badge arc-badge-info" title="Codice di stato HTTP"><?php echo esc_html($log->status_code); ?></span></td>
                                        <td><?php echo esc_html(wp_date(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log->access_time))); ?></td>
                                        <td><span title="<?php echo esc_attr($log->user_agent); ?>"><?php echo esc_html(substr($log->user_agent, 0, 50)) . (strlen($log->user_agent) > 50 ? '...' : ''); ?></span></td>
                                        <td>
                                            <?php if (!empty($log->referrer)) : ?>
                                                <a href="<?php echo esc_url($log->referrer); ?>" target="_blank" rel="noopener noreferrer" title="Apri referrer in nuova scheda">
                                                    <span class="dashicons dashicons-external"></span> <?php echo esc_html(wp_basename($log->referrer)); ?>
                                                </a>
                                            <?php else : ?>
                                                N/D
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>" style="display:inline;">
                                                <input type="hidden" name="action" value="arc_delete_log" />
                                                <input type="hidden" name="log_id" value="<?php echo esc_attr($log->id); ?>" />
                                                <?php wp_nonce_field( 'arc_delete_log_nonce', '_wpnonce' ); ?>
                                                <button type="submit" class="button button-small" title="Elimina questo log" onclick="return confirm('Sei sicuro di voler eliminare questo log?');">
                                                    <span class="dashicons dashicons-trash"></span>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                        <div class="tablenav bottom">
                            <div class="tablenav-pages">
                                <?php
                                $page_links = paginate_links( array(
                                    'base' => add_query_arg( array('paged' => '%#%', 's' => $search_query) ),
                                    'format' => '',
                                    'prev_text' => '&laquo;',
                                    'next_text' => '&raquo;',
                                    'total' => $total_log_pages,
                                    'current' => $current_page,
                                ));
                                if ( $page_links ) {
                                    echo '<span class="displaying-num">' . sprintf( _n( '%s accesso', '%s accessi', $total_logged_redirects, 'auto-redirect-cleaner' ), number_format_i18n( $total_logged_redirects ) ) . '</span>' . $page_links;
                                }
                                ?>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <div class="arc-two-columns">
                <div class="postbox">
                    <h2 class="hndle"><span>Top 5 Redirect Più Usati (ultimi 30 giorni)</span></h2>
                    <div class="inside">
                        <?php if (empty($most_accessed_redirects)) : ?>
                            <p>Nessun dato disponibile.</p>
                        <?php else : ?>
                            <ol>
                                <?php foreach ($most_accessed_redirects as $redirect) : ?>
                                    <li><code><?php echo esc_html($redirect->old_url); ?></code> (<?php echo sprintf(_n('%s accesso', '%s accessi', $redirect->access_count, 'auto-redirect-cleaner'), number_format_i18n($redirect->access_count)); ?>)</li>
                                <?php endforeach; ?>
                            </ol>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="postbox">
                    <h2 class="hndle"><span>Top 5 Redirect Meno Usati (ultimi 30 giorni, con almeno 1 accesso)</span></h2>
                    <div class="inside">
                        <?php if (empty($least_accessed_redirects)) : ?>
                            <p>Nessun dato disponibile.</p>
                        <?php else : ?>
                            <ol>
                                <?php foreach ($least_accessed_redirects as $redirect) : ?>
                                    <li><code><?php echo esc_html($redirect->old_url); ?></code> (<?php echo sprintf(_n('%s accesso', '%s accessi', $redirect->access_count, 'auto-redirect-cleaner'), number_format_i18n($redirect->access_count)); ?>)</li>
                                <?php endforeach; ?>
                            </ol>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <style>
                .arc-stats-cards {
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                    gap: 20px;
                    margin-bottom: 30px;
                }
                .arc-card {
                    background: #fff;
                    border-left: 4px solid #007cba;
                    padding: 20px;
                    box-shadow: 0 1px 1px rgba(0,0,0,.04);
                    display: flex;
                    align-items: center;
                    gap: 15px;
                }
                .arc-card:nth-child(2) { border-left-color: #4caf50; }
                .arc-card:nth-child(3) { border-left-color: #ff9800; }
                .arc-card:nth-child(4) { border-left-color: #f44336; } /* New color for homepage redirects */
                .arc-card-icon {
                    width: 50px;
                    height: 50px;
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                .arc-card-icon .dashicons {
                    font-size: 28px;
                    width: 28px;
                    height: 28px;
                }
                .arc-card-title {
                    margin: 0;
                    font-size: 14px;
                    color: #666;
                }
                .arc-card-number {
                    margin: 0;
                    font-size: 24px;
                    font-weight: bold;
                    color: #333;
                }
                .arc-badge {
                    display: inline-block;
                    padding: 3px 8px;
                    border-radius: 3px;
                    font-size: 11px;
                    font-weight: 600;
                    line-height: 1;
                    text-align: center;
                    white-space: nowrap;
                    vertical-align: baseline;
                    color: #fff;
                }
                .arc-badge-success { background-color: #4CAF50; }
                .arc-badge-warning { background-color: #FFC107; color: #333; }
                .arc-badge-danger { background-color: #F44336; }
                .arc-badge-info { background-color: #2196F3; }
                .arc-two-columns {
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 20px;
                    margin-top: 30px;
                }
                @media (max-width: 782px) {
                    .arc-two-columns {
                        grid-template-columns: 1fr;
                    }
                }
                /* CSS per il padding dei titoli hndle */
                .postbox .hndle {
                    padding: 8px 12px; /* Aggiunge padding ai titoli */
                }
                .arc-sticky-header thead {
                    position: sticky;
                    top: 32px; /* Altezza della barra admin di WordPress */
                    background: #f0f0f1; /* Colore di sfondo per l'header sticky */
                    z-index: 10;
                    box-shadow: 0 2px 2px -1px rgba(0, 0, 0, 0.1);
                }
                .arc-search-bar {
                    margin-bottom: 20px;
                    padding: 10px 0;
                    border-bottom: 1px solid #eee;
                }
                .arc-search-bar form {
                    display: flex;
                    gap: 10px;
                    align-items: center;
                }
                .arc-search-bar input[type="text"] {
                    flex-grow: 1;
                    max-width: 400px;
                }
                .arc-copy-url {
                    cursor: pointer;
                    margin-left: 5px;
                    vertical-align: middle;
                    color: #007cba;
                }
                .arc-copy-url .dashicons {
                    font-size: 16px;
                    width: 16px;
                    height: 16px;
                }
                .arc-copy-url:hover {
                    color: #005b8e;
                }
            </style>
            <script>
                document.addEventListener('DOMContentLoaded', function() {
                    document.querySelectorAll('.arc-copy-url').forEach(button => {
                        button.addEventListener('click', function(e) {
                            e.preventDefault();
                            const urlToCopy = this.dataset.url;
                            navigator.clipboard.writeText(urlToCopy).then(() => {
                                const originalTitle = this.title;
                                this.title = 'Copiato!';
                                setTimeout(() => {
                                    this.title = originalTitle;
                                }, 1500);
                            }).catch(err => {
                                console.error('Errore nella copia dell\'URL: ', err);
                            });
                        });
                    });
                });
            </script>
        </div>
        <?php
    }
}